// /////////////////////////////////////////////////////////////////////////////
// DR DOBB'S CHALLENGES
//
// Filename       : entity.h
// Date           : February 2008
//
// Description    : This file defines the Entity class which is the parent
//                  class of all in-game entities.  All child entities must
//                  be based on this class.
//
//                  Init() must be called at the beginning of game level
//                  to ensure all static states are initialized before
//                  new level entities are created.  ReleaseAll() must be
//                  called at the end of a game level, either when CGame
//                  is deleted or a level is restarted (and therefore,
//                  entities are re-created).
//
// ///////////////////////////////////////////////////////////////////////////
#pragma once



#include <list>

#include <windows.h>

#include "global.h"
#include "sound.h"



class Level;

class Entity 
{

  protected:

    enum EntityEvent
    {
      EE_NONE = 0,
      EE_JUMP,
      EE_LAND,
      EE_BLOCKED_BOTTOM = EE_LAND,
      EE_DIE,
      EE_BLOCKED_LEFT,
      EE_BLOCKED_RIGHT,
      EE_BLOCKED_TOP,
      EE_BLOCKED_CARRIED_MOVE,
      EE_CARRIED,
      EE_DROPPED,
      EE_CARRY,
      EE_DROP,
      EE_INIT,
      EE_MOVED_LEFT,
      EE_MOVED_RIGHT,
      EE_MOVED_UP,
      EE_MOVED_DOWN,
      EE_SLEEP,
      EE_WAKEUP,
      EE_KILLED,
      EE_HURT,
      EE_JUMP_DONE,
    };

    Dobbs::AnimationIndex         m_CurrentAnimation;
    size_t                        m_AnimationFrame;

    float                         m_AnimationFrameTime;

    std::list<Entity*>            m_CarriedEntities;

    Entity*                       m_pCarriedBy;
    Entity*                       m_pPreviousCarrier;
    float                         m_CarryAgainDelay;

    Level*                        m_pLevel;

    bool                          m_UsingAngle;
    float                         m_Angle;

    int                           m_State;
    float                         m_StateTime;

    DWORD                         m_Color;

    Dobbs::EntityTypes            m_Type;
    int                           m_X;
    int                           m_Y;
    float                         m_Z;
    float                         m_VelX;
    float                         m_VelY;
    float                         m_LifeTime;

    float                         m_FractX;
    float                         m_FractY;

    // time to wait before the item can be picked
    float                         m_PickupDelay;

    bool                          m_OnGround;
    bool                          m_Jumping;
    bool                          m_Floating;

    int                           m_MoveCarriedObjectFlags;

    // how long to display whited
    float                         m_HitTime;

    int                           m_FallSpeed;
    int                           m_FallCount;

    int                           m_JumpCount;
    float                         m_JumpVelocity;

    bool                          m_MovedLastFrame;

    // is the entity sleeping (ie outside the current players position)
    bool                          m_Sleeping;
    bool                          m_CanSleep;
    bool                          m_DiesIfOutsideScreen;

    bool                          m_CanCollideWithBackground;

    bool                          m_CanBeCarried;

    int                           m_Health;

    Dobbs::DirFlags               m_Direction;

    // used to signal in the main loop that this entity can be removed
    bool                          m_RemoveMe;


    bool                          m_Hidden;
    bool                          m_Fadeout;

    RECT                          m_CollisionRect;

    POINT                         m_DisplayOffset;


    bool                          CollidesWith( Entity* pOtherEntity );


  public:

    int                           m_SpawnPosX;
    int                           m_SpawnPosY;

    // used for different purposes
    int                           m_ExtraData;
    int                           m_ExtraData2;


    Entity();
    virtual ~Entity();

    static Entity*                SpawnEntity( Dobbs::EntityTypes EType, int X, int Y, Dobbs::DirFlags StartDir );

    virtual void                  SetPosition( int xVal, int yVal );

    int                           GetX();
    int                           GetY();
    int                           GetSpawnX();
    int                           GetSpawnY();
    bool                          CanBePicked() const;
    float                         Angle() const;
    void                          SetAngle( float Angle );

    RECT                          CollisionRect();

    Dobbs::EntityTypes            GetType();

    virtual void                  Hide();
    virtual void                  Unhide();
    virtual bool                  IsHidden();

    virtual void                  Init( bool Editor );
    virtual void                  SetDirection( Dobbs::DirFlags Dir );

    void                          SetVelX( float VelX );
    void                          SetVelY( float VelY );
    void                          SetLifeTime( float LifeTime );

    float                         GetVelX();
    float                         GetVelY();

    virtual void                  Update( Level& aLevel );
    virtual void                  UpdateTimed( Level& aLevel, float ElapsedTime );

    void                          UpdateAnimation( float ElapsedTime );

    virtual void                  SetAlpha( int Alpha );
    void                          SetColor( unsigned int Color );

    virtual void                  Render( int XOffset, int YOffset );

    virtual void                  HandleAnimation();

    bool                          CanBeRemoved() const;

    // trigger auto-fadeout (alpha followed by autoremove)
    void                          Fadeout();


    void                          SetFrame( Dobbs::AnimationIndex Frame );

    int                           Move( Level& aLevel, float DX, float DY, int DirFlags = Dobbs::DIR_NONE );

    bool                          Jump( Level& aLevel, int JumpPower );
    bool                          ForceJump( Level& aLevel, int JumpPower );

    bool                          OnGround() const;

    Dobbs::DirFlags               Direction() const;

    void                          Die();

    virtual void                  OnEvent( const EntityEvent Event, int Param1 = 0, int Param2 = 0, const std::string& TextParam = std::string(), Entity* pEntity = NULL );

    void                          Carry( Entity* pCarried );
    void                          DropOff( Entity* pCarried );
    
    void                          SetState( int NewState );

    virtual bool                  Hit();
    void                          Hurt( int Damage = 1 );

    Entity*                       Carrier() const;
    int                           MoveCarriedObjectFlags() const;

    bool                          CanBeCarried() const;

    friend class Level;

};


